"""
Combined LLM Survey Runner

This script combines functionality to run surveys using different LLM models:
- Llama 3.1 (8B and 70B)
- Gemma 2 (9B and 27B)
- GPT-4
- Gemini Pro

With different prompt settings:
- Basic: Standard PVQ questions
- BWVr: Value-anchored PVQ questions
- Demographic: PVQ questions with demographic profiles
- Names: PVQ questions with different names
- Personas: PVQ questions with different personas

Usage:
    python combined_llm_survey.py --model llama --model_size 8 --setting basic --gender female --temp 0.0
    python combined_llm_survey.py --model gpt4 --setting personas --personas_file personas.csv --temp 0.0
"""

import os
import sys
import argparse
import csv
from datetime import datetime
import random
import re
import time
from tqdm import tqdm
import traceback
from dotenv import load_dotenv

# Load environment variables from .env file
load_dotenv()

# Configure API clients for OpenAI and Google
try:
    import openai
    openai.api_key = os.getenv('OPENAI_API_KEY')
except ImportError:
    print("OpenAI package not installed. GPT-4 functionality will not be available.")

try:
    from google import generativeai
    generativeai.configure(api_key=os.getenv('GOOGLE_API_KEY'))
except ImportError:
    print("Google Generative AI package not installed. Gemini Pro functionality will not be available.")

# Login to Hugging Face
try:
    from huggingface_hub import login
    login(os.getenv('HF_TOKEN'))
except ImportError:
    print("Hugging Face Hub package not installed. Llama/Gemma functionality will not be available.")

class ValueDescriptions:
    def __init__(self):
        # Basic value descriptions (PVQ) - Male version
        self.descriptions_male = [
            "It is important to him to form his views independently.",
            "It is important to him that his country is secure and stable.",
            "It is important to him to have a good time.",
            "It is important to him to avoid upsetting other people.",
            "It is important to him that the weak and vulnerable in society be protected.",
            "It is important to him that people do what he says they should.",
            "It is important to him never to think he deserves more than other people.",
            "It is important to him to care for nature.",
            "It is important to him that no one should ever shame him.",
            "It is important to him always to look for different things to do.",
            "It is important to him to take care of people he is close to.",
            "It is important to him to have the power that money can bring.",
            "It is very important to him to avoid disease and protect his health.",
            "It is important to him to be tolerant toward all kinds of people and groups.",
            "It is important to him never to violate rules or regulations.",
            "It is important to him to make his own decisions about his life.",
            "It is important to him to have ambitions in life.",
            "It is important to him to maintain traditional values and ways of thinking.",
            "It is important to him that people he knows have full confidence in him.",
            "It is important to him to be wealthy.",
            "It is important to him to take part in activities to defend nature.",
            "It is important to him never to annoy anyone.",
            "It is important to him to develop his own opinions.",
            "It is important to him to protect his public image.",
            "It is very important to him to help the people dear to him.",
            "It is important to him to be personally safe and secure.",
            "It is important to him to be a dependable and trustworthy friend.",
            "It is important to him to take risks that make life exciting.",
            "It is important to him to have the power to make people do what he wants.",
            "It is important to him to plan his activities independently.",
            "It is important to him to follow rules even when no-one is watching.",
            "It is important to him to be very successful.",
            "It is important to him to follow his family's customs or the customs of a religion.",
            "It is important to him to listen to and understand people who are different from him.",
            "It is important to him to have a strong state that can defend its citizens.",
            "It is important to him to enjoy life's pleasures.",
            "It is important to him that every person in the world have equal opportunities in life.",
            "It is important to him to be humble.",
            "It is important to him to figure things out himself.",
            "It is important to him to honor the traditional practices of his culture.",
            "It is important to him to be the one who tells others what to do.",
            "It is important to him to obey all the laws.",
            "It is important to him to have all sorts of new experiences.",
            "It is important to him to own expensive things that show his wealth.",
            "It is important to him to protect the natural environment from destruction or pollution.",
            "It is important to him to take advantage of every opportunity to have fun.",
            "It is important to him to concern himself with every need of his dear ones.",
            "It is important to him that people recognize what he achieves.",
            "It is important to him never to be humiliated.",
            "It is important to him that his country protect itself against all threats.",
            "It is important to him never to make other people angry.",
            "It is important to him that everyone be treated justly, even people he doesn't know.",
            "It is important to him to avoid anything dangerous.",
            "It is important to him to be satisfied with what he has and not ask for more.",
            "It is important to him that all his friends and family can rely on him completely.",
            "It is important to him to be free to choose what he does by himself.",
            "It is important to him to accept people even when he disagrees with them."
        ]
        
        # Basic value descriptions (PVQ) - Female version
        self.descriptions_female = [
            "It is important to her to form her views independently.",
            "It is important to her that her country is secure and stable.",
            "It is important to her to have a good time.",
            "It is important to her to avoid upsetting other people.",
            "It is important to her that the weak and vulnerable in society be protected.",
            "It is important to her that people do what she says they should.",
            "It is important to her never to think she deserves more than other people.",
            "It is important to her to care for nature.",
            "It is important to her that no one should ever shame her.",
            "It is important to her always to look for different things to do.",
            "It is important to her to take care of people she is close to.",
            "It is important to her to have the power that money can bring.",
            "It is very important to her to avoid disease and protect her health.",
            "It is important to her to be tolerant toward all kinds of people and groups.",
            "It is important to her never to violate rules or regulations.",
            "It is important to her to make her own decisions about her life.",
            "It is important to her to have ambitions in life.",
            "It is important to her to maintain traditional values and ways of thinking.",
            "It is important to her that people she knows have full confidence in her.",
            "It is important to her to be wealthy.",
            "It is important to her to take part in activities to defend nature.",
            "It is important to her never to annoy anyone.",
            "It is important to her to develop her own opinions.",
            "It is important to her to protect her public image.",
            "It is very important to her to help the people dear to her.",
            "It is important to her to be personally safe and secure.",
            "It is important to her to be a dependable and trustworthy friend.",
            "It is important to her to take risks that make life exciting.",
            "It is important to her to have the power to make people do what she wants.",
            "It is important to her to plan her activities independently.",
            "It is important to her to follow rules even when no-one is watching.",
            "It is important to her to be very successful.",
            "It is important to her to follow her family's customs or the customs of a religion.",
            "It is important to her to listen to and understand people who are different from her.",
            "It is important to her to have a strong state that can defend its citizens.",
            "It is important to her to enjoy life's pleasures.",
            "It is important to her that every person in the world have equal opportunities in life.",
            "It is important to her to be humble.",
            "It is important to her to figure things out herself.",
            "It is important to her to honor the traditional practices of her culture.",
            "It is important to her to be the one who tells others what to do.",
            "It is important to her to obey all the laws.",
            "It is important to her to have all sorts of new experiences.",
            "It is important to her to own expensive things that show her wealth.",
            "It is important to her to protect the natural environment from destruction or pollution.",
            "It is important to her to take advantage of every opportunity to have fun.",
            "It is important to her to concern herself with every need of her dear ones.",
            "It is important to her that people recognize what she achieves.",
            "It is important to her never to be humiliated.",
            "It is important to her that her country protect itself against all threats.",
            "It is important to her never to make other people angry.",
            "It is important to her that everyone be treated justly, even people she doesn't know.",
            "It is important to her to avoid anything dangerous.",
            "It is important to her to be satisfied with what she has and not ask for more.",
            "It is important to her that all her friends and family can rely on her completely.",
            "It is important to her to be free to choose what she does by herself.",
            "It is important to her to accept people even when she disagrees with them."
        ]
        
        # BWVr values
        self.bwvr_values = [
            "developing your own original ideas and opinions",
            "being free to act independently",
            "having an exciting life",
            "having all sorts of new experiences",
            "taking advantage of every opportunity to enjoy life's pleasures",
            "being ambitious and successful",
            "having the power that money and possessions can bring",
            "having the authority to get others to do what you want",
            "protecting your public image and avoiding being shamed",
            "living and acting in ways that ensure that you are personally safe and secure",
            "living in a safe and stable society",
            "following cultural, family or religious practices",
            "obeying all rules and laws",
            "making sure you never upset or annoy others",
            "being humble and avoiding public recognition",
            "being a completely dependable and trustworthy friend and family member",
            "helping and caring for the wellbeing of those who are close",
            "caring and seeking justice for everyone, especially the weak and vulnerable in society",
            "protecting the natural environment from destruction or pollution",
            "caring for the welfare of animals",
            "being open-minded and accepting of people and ideas, even when you disagree with them"
        ]

class BaseSetting:
    def __init__(self, args):
        self.args = args
        self.value_descriptions = ValueDescriptions()
    
    def write_first_csv_row(self, csvwriter):
        pass
    
    def get_identifier(self, iteration):
        return None
    
    def choose_descriptions(self, args, identifier):
        pass
    
    def generate_prompt(self, identifier, description_text):
        pass
    
    def write_csv_row(self, args, csvwriter, date_str, iteration, index, description, rating, identifier):
        pass

class BasicSetting(BaseSetting):
    def write_first_csv_row(self, csvwriter):
        csvwriter.writerow(["Date", "Gender", "Iteration", "Statement", "Rating"])
    
    def get_identifier(self, iteration):
        return None
    
    def choose_descriptions(self, args, identifier):
        if args.gender == "male":
            return self.value_descriptions.descriptions_male
        else:
            return self.value_descriptions.descriptions_female
    
    def generate_prompt(self, identifier, description_text):
        return (f"For each of the following descriptions, please answer how much the person described is like you "
                f"from 1 (Not like me at all) to 6 (Very much like me), without elaborating on your reasoning.\n\n"
                f"Descriptions:\n{description_text}")
    
    def write_csv_row(self, args, csvwriter, date_str, iteration, index, description, rating, identifier):
        statement_label = f"PVQ{index+1}"
        csvwriter.writerow([date_str, args.gender, iteration+1, statement_label, rating])

class BwvrSetting(BaseSetting):
    def write_first_csv_row(self, csvwriter):
        csvwriter.writerow(["Date", "Gender", "Iteration", "Selected_Value", "Statement", "Rating"])
    
    def get_identifier(self, iteration):
        bwvr_index = iteration % len(self.value_descriptions.bwvr_values)
        return self.value_descriptions.bwvr_values[bwvr_index]
    
    def choose_descriptions(self, args, identifier):
        if args.gender == "male":
            return self.value_descriptions.descriptions_male
        else:
            return self.value_descriptions.descriptions_female
    
    def generate_prompt(self, identifier, description_text):
        return (f"For each of the following descriptions, please answer how much the person described is like you "
                f"from 1 (Not like me at all) to 6 (Very much like me), without elaborating on your reasoning. "
                f"Answer as a person that values {identifier}.\n\n"
                f"Descriptions:\n{description_text}")
    
    def write_csv_row(self, args, csvwriter, date_str, iteration, index, description, rating, identifier):
        statement_label = f"PVQ{index+1}"
        csvwriter.writerow([date_str, args.gender, iteration+1, identifier, statement_label, rating])

class DemographicSetting(BaseSetting):
    def __init__(self, args):
        super().__init__(args)
        # Define demographic variables directly from the provided file
        self.age_list = list(range(18, 76))
        self.genders = ["male", "female", "non-binary gender", "other"]
        self.occupation_list = [
            "professional and technical expertise",
            "higher administrative roles",
            "clerical work",
            "sales",
            "service",
            "skilled work",
            "semi-skilled work",
            "unskilled work",
            "farm work",
            "farm management or ownership",
            "no prior job experience"
        ]
        self.hobby_list = [
            "shopping", "driving", "taking care of pets", "managing financial matters", "taking a rest",
            "going to the hairdresser / barber", "childcare / babysitting", "preparing a hot drink",
            "conducting personal care", "conducting personal business", "taking care of others",
            "cleaning/ fixing things", "talking on the telephone", "creative writing / keeping a journal",
            "knitting / needlecrafts", "playing table games", "going to watch a sports event",
            "cooking / baking as a hobby", "doing puzzles / crosswords", "using a computer",
            "taking photographs", "reading a religious book", "written communications",
            "looking at photo albums / home videos", "researching family / local history",
            "reading a newspaper / magazine", "watching nature", "playing bingo", "watching television",
            "listening to the radio / music", "relaxing / meditating", "entering competitions",
            "reading a book", "flower arranging", "going to the beach", "dancing", "swimming",
            "playing a ball game", "walking", "hiking / rambling", "exercising", "riding a bicycle",
            "going on holiday / travelling", "attending a leisure / social group", "going to gardens / parks",
            "fishing", "having a picnic / BBQ", "spending time with family / friends", "eating out",
            "going to parties", "going for drinks at pubs / social clubs", "volunteer work",
            "cultural visits", "going to music / performing arts events", 
            "going to church / mosque / synagogue / temple / other",
            "collecting (stamps, posters, figures)", "drawing / painting", "interior decorating"
        ]
    
    def write_first_csv_row(self, csvwriter):
        csvwriter.writerow(["Date", "Iteration", "Number", "Description", "Rating", "Demographic Characteristics"])
    
    def get_identifier(self, iteration):
        # Randomly select demographic characteristics
        selected_age = random.choice(self.age_list)
        selected_gender = random.choice(self.genders)
        selected_occupation = random.choice(self.occupation_list)
        selected_hobby = random.choice(self.hobby_list)
        
        return {
            "age": selected_age,
            "gender": selected_gender,
            "occupation": selected_occupation,
            "hobby": selected_hobby
        }
    
    def choose_descriptions(self, args, identifier):
        if identifier["gender"] == "male":
            return self.value_descriptions.descriptions_male
        else:
            return self.value_descriptions.descriptions_female
    
    def generate_prompt(self, identifier, description_text):
        demographic_str = (f"You are a {identifier['age']} years old that identifies as a {identifier['gender']}, working in the field of "
                          f"{identifier['occupation']}, and likes {identifier['hobby']}.")
        
        return (f"{demographic_str}\n\nFor each of the following descriptions, please answer how much the person described "
                f"is like you from 1 (Not like me at all) to 6 (Very much like me), without elaborating on your reasoning.\n\n"
                f"Descriptions:\n{description_text}")
    
    def write_csv_row(self, args, csvwriter, date_str, iteration, index, description, rating, identifier):
        statement_label = f"PVQ{index+1}"
        demographics_str = f"{identifier['age']} years old that identifies as a {identifier['gender']}, working in the field of {identifier['occupation']}, and likes {identifier['hobby']}"
        csvwriter.writerow([
            date_str, 
            iteration+1, 
            statement_label, 
            description, 
            rating,
            demographics_str
        ])

class NamesSetting(BaseSetting):
    def __init__(self, args):
        super().__init__(args)
        # Define titles
        self.titles = ["Mr.", "Ms.", "Mx."]
        
        # Define names from different ethnic backgrounds (based on Aher et al., 2023)
        self.american_indian_names = [
            'Tsinnie', 'Shije', 'Gishie', 'Laughing', 'Whitehat', 'Hosteen', 'Nez', 'Begaye', 'Tsosie', 'Twobulls',
            'Etsitty', 'Begay', 'Clah', 'Delgarito', 'Tapaha', 'Yazzie', 'Goseyun', 'Haskie', 'Keams', 'Altaha',
            'Becenti', 'Tsinnijinnie', 'Yellowhair', 'Todacheene', 'Begaye', 'Tabaha', 'Apachito', 'Littlelight', 'Goldtooth',
            'Cayaditto', 'Kanuho', 'Roanhorse', 'Notah', 'Chasinghawk', 'Todacheenie', 'Blueeyes', 'Blackgoat', 'Smallcanyon',
            'Secatero', 'Secody', 'Atcitty', 'Ganadonegro', 'Manuelito', 'Manygoats', 'Bitsuie', 'Wauneka', 'Chinana',
            'Whiteplume', 'Henio', 'Cosay', 'Peshlakai', 'Clitso', 'Roanhorse', 'Henio', 'Cosay', 'Peshlakai', 'Clitso'
        ]
        
        self.asian_pacific_names = [
            'Ho', 'Vu', 'Yang', 'Tran', 'Ng', 'Le', 'Cheng', 'Wang', 'Nguyen', 'Kim',
            'Pham', 'Liu', 'Lin', 'Shin', 'Xiong', 'Zhao', 'Zhang', 'Sharma', 'Hu', 'Choi',
            'Yu', 'Sun', 'Chu', 'Thao', 'Ha', 'Chiu', 'Gupta', 'Li', 'Jain', 'Jiang',
            'Yan', 'Kang', 'Lo', 'Shen', 'Ngo', 'Cheung', 'Shen', 'Chung', 'Lai', 'Chen',
            'Ko', 'Huang', 'Oh', 'Trinh', 'Shin', 'Tam', 'Yoon', 'Luu', 'Thai', 'Zhou',
            'Song', 'Duong', 'Chau', 'Moua'
        ]
        
        self.black_names = [
            'Smalls', 'Jalloh', 'Abdullahi', 'Jeanpierre', 'Osei', 'Jama', 'Jeancharles', 'Fofana', 'Calixte', 'Jeanjacques',
            'Cisse', 'Jeanlouis', 'Sesay', 'Kebede', 'Jeanbaptiste', 'Manigault', 'Wigfall', 'Mondesir', 'Gadson', 'Jeanfrancois',
            'Koroma', 'Mwangi', 'Louissaint', 'Prioleau', 'Grandberry', 'Straughter', 'Mekonnen', 'Conteh', 'Bekele', 'Traore',
            'Diop', 'Njoroge'
        ]
        
        self.hispanic_names = [
            'Rodriguez', 'Chavez', 'Lopez', 'Molina', 'Vasquez', 'Medina', 'Calderon', 'Sanchez', 'Herrera', 'Rojas',
            'Vargas', 'Gomez', 'Ortega', 'Garcia', 'Pena', 'Ramirez', 'Rios', 'Jimenez', 'Serrano', 'Vazquez',
            'Morales', 'Fuentes', 'Carrillo', 'Moreno', 'Castillo', 'Munoz', 'Gonzalez', 'Salinas', 'Mejia', 'Alvarez',
            'Guerrero', 'Padilla', 'Diaz', 'Ruiz', 'Perez', 'Flores', 'Contreras', 'Soto', 'Marquez', 'Espinoza',
            'Torres', 'Salazar', 'Nunez', 'Aguilar', 'Vega', 'Rivera', 'Delgado', 'Avila', 'Aguirre', 'Rivas'
        ]
        
        self.white_names = [
            'Snyder', 'Foley', 'Conrad', 'Brennan', 'Hoffman', 'Nielsen', 'Moyer', 'Flynn', 'Rasmussen', 'Schultz',
            'Christensen', 'McMahon', 'Weber', 'Mccarthy', 'Morse', 'Schneider', 'Russo', 'Case', 'Oneill', 'Hartman',
            'Howe', 'Roth', 'Bauer', 'Parsons', 'Peck', 'Koch', 'Hoover', 'Macdonald', 'Schaefer', 'Hebert',
            'Berg', 'Mcmahon', 'Stark', 'Kramer', 'Reilly', 'Mayer', 'Hensley', 'Duffy', 'Kline', 'Donovan',
            'Friedman', 'Krueger', 'Meyer', 'Brandt', 'Boyle', 'Gallagher', 'Olsen', 'Schmidt', 'Huber', 'Larson',
            'Johnston', 'Weiss', 'Owen', 'Carlson'
        ]
        
        # Create a combined list of all unique names
        all_names = list(set(
            self.american_indian_names + 
            self.asian_pacific_names + 
            self.black_names + 
            self.hispanic_names + 
            self.white_names
        ))
        
        # Subset the names to ensure we have enough
        self.all_names_subset = random.sample(all_names, min(300, len(all_names)))
    
    def write_first_csv_row(self, csvwriter):
        csvwriter.writerow(["Date", "Iteration", "Number", "Description", "Rating", "Titles & Names"])
    
    def get_identifier(self, iteration):
        # Select a title randomly for this iteration
        selected_title = random.choice(self.titles)
        
        # Make a copy of the names subset to ensure uniqueness within this iteration
        current_names_subset = self.all_names_subset.copy()
        random.shuffle(current_names_subset)
        
        # Choose a name
        selected_name = random.choice(current_names_subset)
        
        return {
            "title": selected_title,
            "name": selected_name,
            "gender": "male" if selected_title == "Mr." else "female" if selected_title == "Ms." else random.choice(["male", "female"])
        }
    
    def choose_descriptions(self, args, identifier):
        if identifier["gender"] == "male":
            return self.value_descriptions.descriptions_male
        else:
            return self.value_descriptions.descriptions_female
    
    def generate_prompt(self, identifier, description_text):
        return (f"For each of the following descriptions, please answer how much the person described is like you "
                f"from 1 (Not like me at all) to 6 (Very much like me), without elaborating on your reasoning. "
                f"Answer as {identifier['title']} {identifier['name']}.\n\n"
                f"Descriptions:\n{description_text}")
    
    def write_csv_row(self, args, csvwriter, date_str, iteration, index, description, rating, identifier):
        statement_label = f"PVQ{index+1}"
        names_str = f"{identifier['title']} {identifier['name']}"
        csvwriter.writerow([
            date_str, 
            iteration+1, 
            statement_label, 
            description, 
            rating,
            names_str
        ])

class PersonasSetting(BaseSetting):
    def __init__(self, args):
        super().__init__(args)
        # Load personas from CSV file
        self.personas = {}
        self._load_personas_csv(args)
    
    def _load_personas_csv(self, args):
        """Load personas from the CSV file"""
        # Get the personas file path
        personas_csv = os.path.join(args.output_dir, args.personas_file)
        
        if not os.path.exists(personas_csv):
            raise FileNotFoundError(f"Personas CSV file not found: {personas_csv}")
        
        try:
            with open(personas_csv, 'r', newline='', encoding='utf-8') as csvfile:
                reader = csv.reader(csvfile)
                headers = next(reader)  # Get headers
                
                # Handle BOM character if present
                if headers[0].startswith('\ufeff'):
                    headers[0] = headers[0][1:]
                
                # Determine column indices
                persona_idx = headers.index('Persona') if 'Persona' in headers else 0
                desc_idx = headers.index('Description') if 'Description' in headers else 1
                
                # Read all personas
                for row in reader:
                    if len(row) > 1:  # Ensure row has enough elements
                        persona_num = row[persona_idx]
                        persona_desc = row[desc_idx]
                        self.personas[persona_num] = persona_desc
            
            print(f"Loaded {len(self.personas)} personas from {personas_csv}")
            
            if len(self.personas) < 10:
                print("Warning: Found fewer than 10 personas. Make sure the file is correctly formatted.")
        
        except Exception as e:
            raise Exception(f"Error loading personas from CSV: {str(e)}")
    
    def write_first_csv_row(self, csvwriter):
        csvwriter.writerow(["Date", "Persona Number", "Number", "Description", "Rating", "Persona Text"])
    
    def get_identifier(self, iteration):
        # Get list of persona numbers and cycle through them
        persona_nums = list(self.personas.keys())
        if not persona_nums:
            raise ValueError("No personas loaded")
            
        persona_idx = iteration % len(persona_nums)
        persona_num = persona_nums[persona_idx]
        persona_text = self.personas[persona_num]
        
        # Determine gender based on text clues
        gender = self._determine_gender_from_text(persona_text)
        
        return {
            "persona_number": persona_num,
            "persona_text": persona_text,
            "gender": gender
        }
    
    def _determine_gender_from_text(self, text):
        """Determine gender based on text clues in the persona description"""
        text_lower = text.lower()
        
        if "she" in text_lower or "her" in text_lower or "mother" in text_lower or "mom" in text_lower:
            return "female"
        elif "he" in text_lower or "him" in text_lower or "father" in text_lower or "dad" in text_lower or text.find("John") != -1 or text.find("Michael") != -1 or text.find("James") != -1 or text.find("Alex") != -1 or text.find("Mark") != -1:
            return "male"
        else:
            # Default to female if gender is ambiguous
            return "female"
    
    def choose_descriptions(self, args, identifier):
        # Choose descriptions based on detected gender
        if identifier["gender"] == "male":
            return self.value_descriptions.descriptions_male
        else:
            return self.value_descriptions.descriptions_female
    
    def generate_prompt(self, identifier, description_text):
        return (f"For each of the following descriptions, please answer how much the person described is like you "
                f"from 1 (Not like me at all) to 6 (Very much like me), without elaborating on your reasoning. "
                f"Answer as Persona: {identifier['persona_text']}.\n\n"
                f"Descriptions:\n{description_text}")
    
    def write_csv_row(self, args, csvwriter, date_str, iteration, index, description, rating, identifier):
        statement_label = f"PVQ{index+1}"
        csvwriter.writerow([
            date_str, 
            identifier["persona_number"], 
            statement_label, 
            description, 
            rating,
            identifier["persona_text"]
        ])

def extract_numbers(input_string):
    """Extract numbers from a string"""
    return ''.join(char for char in input_string if char.isdigit())

def parse_ratings(response, expected_count):
    """Parse ratings from model response, similar to parse_result in the original code"""
    print("\nRaw response to parse:")
    print(response)
    
    raw_rating = response.split('\n')
    final_rating = []
    
    print(f"\nFound {len(raw_rating)} lines in response")
    
    for rating in raw_rating:
        if re.match(r'(\d)+.*', rating):
            try:
                # Extract the last word which should be the rating
                rating_parts = rating.strip().split(' ')
                rating_value = rating_parts[-1]
                # Extract only digits
                rating_value = extract_numbers(rating_value)
                int_rating = int(rating_value)
                
                if 1 <= int_rating <= 6:
                    final_rating.append(int_rating)
                    print(f"Found valid rating: {int_rating}")
                else:
                    print(f"Rating out of range: {int_rating}")
            except Exception as e:
                print(f"Error parsing rating from line: {rating}")
                continue
    
    print(f"\nExtracted {len(final_rating)} valid ratings (expected {expected_count})")
    print(final_rating)
    
    # If we didn't get enough ratings, try a more lenient approach
    if len(final_rating) != expected_count:
        print("\nTrying alternate parsing method...")
        final_rating = []
        
        for line in raw_rating:
            try:
                # Look for any digit between 1-6 in the line
                for word in line.split():
                    if word.isdigit() and 1 <= int(word) <= 6:
                        final_rating.append(int(word))
                        print(f"Found rating: {word}")
                        break
            except:
                continue
        
        print(f"Alternate method found {len(final_rating)} ratings")
    
    return final_rating

def get_completion_llama_gemma(model_name, model_size, prompt, temperature):
    """Get completion from Llama or Gemma models"""
    try:
        import transformers
        import torch
        
        # Select the correct model ID
        if model_name == "llama":
            if model_size == 8:
                model_id = "meta-llama/Meta-Llama-3.1-8B-Instruct"
            elif model_size == 70:
                model_id = "meta-llama/Meta-Llama-3.1-70B-Instruct"
            else:
                raise ValueError(f"Unsupported Llama model size: {model_size}")
        else:  # gemma
            if model_size == 9:
                model_id = "google/gemma-2-9b-it"
            elif model_size == 27:
                model_id = "google/gemma-2-27b-it"
            else:
                raise ValueError(f"Unsupported Gemma model size: {model_size}")
        
        print(f"Loading model: {model_id}")
        
        # Create the pipeline
        generator = transformers.pipeline(
            "text-generation",
            model=model_id,
            device_map="auto",
            torch_dtype=torch.bfloat16
        )
        
        # Prepare messages
        messages = [
            {"role": "user", "content": prompt},
        ]
        
        print(f"Generating response with temperature={temperature}")
        
        # Set up generation parameters based on temperature
        if temperature == 0:
            outputs = generator(
                messages,
                do_sample=False,
                top_p=None,
                temperature=None,
                max_new_tokens=2048
            )
        else:
            outputs = generator(
                messages,
                do_sample=True,
                temperature=temperature,
                max_new_tokens=2048
            )
            
        # Extract the generated text
        result = outputs[0]["generated_text"][-1]["content"]
        print(f"Generated response of length {len(result)}")
        return result
    except Exception as e:
        print(f"Error with {model_name}: {str(e)}")
        traceback.print_exc()
        return None

def get_completion_gpt4(prompt, temperature):
    """Get completion from GPT-4"""
    try:
        print("Sending request to OpenAI API")
        response = openai.chat.completions.create(
            model="gpt-4",
            messages=[
                {"role": "user", "content": prompt}
            ],
            temperature=temperature,
            max_tokens=1000
        )
        result = response.choices[0].message.content
        print(f"Received response from OpenAI API, length: {len(result)}")
        return result
    except Exception as e:
        print(f"Error with GPT-4: {str(e)}")
        traceback.print_exc()
        return None

def get_completion_gemini(prompt, temperature):
    """Get completion from Gemini Pro"""
    try:
        print("Sending request to Gemini API")
        model = generativeai.GenerativeModel('gemini-pro')
        generation_config = generativeai.GenerationConfig(
            temperature=temperature
        )
        response = model.generate_content(
            prompt,
            generation_config=generation_config
        )
        result = response.text
        print(f"Received response from Gemini API, length: {len(result)}")
        return result
    except Exception as e:
        print(f"Error with Gemini Pro: {str(e)}")
        traceback.print_exc()
        return None

def get_completion(args, prompt):
    """Get completion from specified model"""
    if args.model == "llama":
        return get_completion_llama_gemma(args.model, args.model_size, prompt, args.temp)
    elif args.model == "gemma":
        return get_completion_llama_gemma(args.model, args.model_size, prompt, args.temp)
    elif args.model == "gpt4":
        return get_completion_gpt4(prompt, args.temp)
    elif args.model == "gemini":
        return get_completion_gemini(prompt, args.temp)
    else:
        raise ValueError(f"Unknown model: {args.model}")

def get_setting(args):
    """Get the appropriate setting based on the argument"""
    if args.setting == "basic":
        return BasicSetting(args)
    elif args.setting == "bwvr":
        return BwvrSetting(args)
    elif args.setting == "names":
        return NamesSetting(args)
    elif args.setting == "demographic":
        return DemographicSetting(args)
    elif args.setting == "personas":
        return PersonasSetting(args)
    else:
        raise ValueError(f"Unknown setting: {args.setting}")

def run_survey(args):
    """Run the survey with the specified settings"""
    # Get the appropriate setting
    setting = get_setting(args)
    
    # Create results directory based on model
    if args.model == "llama":
        results_path = f'{args.output_dir}/results/{args.model}_{args.model_size}b_results'
        model_str = f"{args.model}{args.model_size}"
    elif args.model == "gemma":
        results_path = f'{args.output_dir}/results/{args.model}_{args.model_size}b_results' 
        model_str = f"{args.model}{args.model_size}"
    elif args.model == "gpt4":
        results_path = f'{args.output_dir}/results/{args.model}_results'
        model_str = args.model
    elif args.model == "gemini":
        results_path = f'{args.output_dir}/results/{args.model}_pro_results'
        model_str = f"{args.model}_pro"
    
    os.makedirs(results_path, exist_ok=True)
    
    # Format temperature string for filename
    temp_str = str(args.temp).replace('.', '')
    
    # Create CSV filename
    if args.setting in ['basic', 'bwvr']:
        csv_filename = f"{results_path}/{args.setting}_{model_str}_{args.gender}_{temp_str}.csv"
        required_iterations = 150
    else:
        csv_filename = f"{results_path}/{args.setting}_{model_str}_{temp_str}.csv"
        required_iterations = 300
    
    # Check if file already exists
    if os.path.exists(csv_filename) and not args.force_overwrite:
        raise FileExistsError(f"Results file already exists: {csv_filename}. Use --force_overwrite to overwrite.")

    print(f"Survey configuration:")
    print(f"- Model: {args.model}{' ' + str(args.model_size) if args.model_size else ''}")
    print(f"- Setting: {args.setting}")
    print(f"- Temperature: {args.temp}")
    print(f"- Output file: {csv_filename}")
    print(f"- Required iterations: {required_iterations}")

    # Open the CSV file 
    with open(csv_filename, "w", newline="", encoding="utf-8") as csvfile:
        csvwriter = csv.writer(csvfile)
        
        # Write header
        setting.write_first_csv_row(csvwriter)
        
        # Get current date string
        date_str = datetime.now().strftime("%x")
        
        # Run the iterations
        for i in tqdm(range(required_iterations)):
            success = False
            retries = 1
            
            # Get identifier and descriptions
            identifier = setting.get_identifier(i)
            descriptions = setting.choose_descriptions(args, identifier)
            
            # Generate prompt
            description_text = "\n".join(f"{j+1}. {desc}" for j, desc in enumerate(descriptions))
            prompt = setting.generate_prompt(identifier, description_text)
            
            while not success and retries <= 10:
                try:
                    # Get model response
                    print(f"\nIteration {i+1}/{required_iterations}")
                    if identifier:
                        print(f"Identifier: {identifier}")
                    
                    # Get response from model
                    response = get_completion(args, prompt)
                    if response is None:
                        raise ValueError("Failed to get response from model")
                    
                    # Print the response
                    print(f"Answers:\n{response}\n")
                    
                    # Parse ratings
                    ratings = parse_ratings(response, len(descriptions))
                    
                    # Validate ratings
                    if not all(1 <= r <= 6 for r in ratings):
                        raise ValueError("Error with model rating - values out of range (1-6)")
                    if len(ratings) != len(descriptions):
                        raise ValueError(f"Got {len(ratings)} ratings, expected {len(descriptions)}")
                    
                    # Write valid results
                    for index, rating in enumerate(ratings):
                        setting.write_csv_row(args, csvwriter, date_str, i, index, descriptions[index], rating, identifier)
                    
                    # Flush after each successful iteration
                    csvfile.flush()
                    success = True
                    print(f"✓ Successfully completed iteration {i+1}")
                    
                except Exception as e:
                    print(f"✗ Error in iteration {i+1}: {str(e)}")
                    retries += 1
                    if retries > 10:
                        raise Exception(f"Failed to get valid response after 10 retries for iteration {i+1}")
                
                # Sleep to respect rate limits (especially important for API-based models)
                sleep_time = 1
                if args.model in ["gpt4", "gemini"]:
                    sleep_time = 3  # Longer sleep for API models to respect rate limits
                print(f"Waiting {sleep_time}s before next request...")
                time.sleep(sleep_time)
    
    print(f"\n✓ Survey completed successfully!")
    print(f"✓ Results saved to: {csv_filename}")

def main():
    """Main entry point for the script"""
    parser = argparse.ArgumentParser(description='Run LLM value survey with multiple models and prompt types')
    
    # Model selection
    parser.add_argument('--model', choices=['llama', 'gemma', 'gpt4', 'gemini'], required=True,
                      help='Model type to use (llama, gemma, gpt4, gemini)')
    parser.add_argument('--model_size', type=int, 
                      choices=[8, 70, 9, 27],
                      help='Model size in billions: Llama (8, 70), Gemma (9, 27)')
    
    # Experiment settings
    parser.add_argument('--setting', choices=['basic', 'bwvr', 'demographic', 'personas', 'names'], required=True,
                      help='The type of prompt setting to run')
    parser.add_argument('--gender', choices=['male', 'female'], default='female',
                      help='Gender for basic and bwvr prompts (male or female)')
    parser.add_argument('--temp', type=float, choices=[0.0, 0.7], default=0.0,
                      help='Temperature for model (0.0 for deterministic, 0.7 for more diverse outputs)')
    
    # Output settings
    parser.add_argument('--output_dir', type=str, default='.',
                      help='Directory for output files')
    parser.add_argument('--force_overwrite', action='store_true',
                      help='Overwrite existing output file if it exists')
    parser.add_argument('--personas_file', type=str, default='personas_gemini.csv',
                      help='CSV file containing personas for the personas setting')
    
    args = parser.parse_args()
    
    # Set default model size if not provided
    if args.model_size is None:
        if args.model == 'llama':
            args.model_size = 8  # Default to 8B for Llama
        elif args.model == 'gemma':
            args.model_size = 9  # Default to 9B for Gemma
    
    # Validate model size is provided for models that need it
    if args.model in ['llama', 'gemma'] and args.model_size is None:
        parser.error(f"--model_size is required for {args.model}")
    
    # Create output directory if it doesn't exist
    os.makedirs(args.output_dir, exist_ok=True)
    
    try:
        run_survey(args)
    except Exception as e:
        print(f"Error running survey: {str(e)}")
        traceback.print_exc()
        return 1
    
    return 0

if __name__ == "__main__":
    sys.exit(main())